// TuningMap.pas: Implementation of the class CTuningMap.
//
// This is the Object Pascal/Delphi version, done by
// Tobias Fleischer alias Tobybear (tobybear@web.de)
// Original C++ version by Mark Henning
//
// (C)opyright in 2003 by Mark Henning, Germany
// Contact email: info@anamark.de or mh@homolog.de
//
// You may use this code for free. If you find an error or make some
// interesting changes, please let me know.
//
// This class deals with the reading/writing of AnaMark / VAZ 1.5 Plus
// tuning files. Be carefull with changes of the functions
// WriteToFile/ReadFromFile because this may lead to incompatibilities!
//
// I think, the source-code is rather self-explaining.
//
// The specifications of the AnaMark / VAZ 1.5 Plus tuning file format
// can be found at http://www.anamark.de
//
// Have fun!
//
//////////////////////////////////////////////////////////////////////

unit TuningMap;
interface
type CTuningMap=class
public
 constructor Create;
 destructor Destroy;override;
 procedure Reset;
 function WriteToFile(szFilepath:string;bSaveBaseFreq:boolean=false):boolean;
 function ReadFromFile(szFilepath:string):boolean;
 function GetBaseFreq:double; // BaseFreq in Hz
 function GetNoteFreq(nNoteIndex:integer):double; // Absolute tune in Hz
 function GetRelativeTune(nNoteIndex:integer):double; // Relative tune in cents
 function SetBaseFreq(dblBaseFreq:double):boolean; // BaseFreq in Hz
 function SetRelativeTune(nNoteIndex:integer;dblTune:double):boolean; // Relative tune in cents

 // Call this directly, when one of the above functions failed:
 function GetLastError:string;
private
 m_dblTunes:array[0..127] of double;	// Unit: Cents
 m_dblBaseFreq:double;    		// Unit: Hz
 m_szErrorString:string;
end;
type eSection=(SEC_None,SEC_Unknown,SEC_Tuning,SEC_ExactTuning);

implementation
uses math,classes,sysutils,windows,dialogs;

function StripBlanks(s:string):string;
var j:integer;
begin
 if s='' then exit;
 j:=1;while ((s[j]<>#0) and ((s[j]=' ') or (s[j]=#9))) do inc(j);
 s:=copy(s,j,length(s)-j+1);
 j:=length(s);
 while ((j>1) and ((s[j]=' ') or (s[j] = #9)) ) do dec(j);
 result:=copy(s,1,j);
end;

{ CTuningMap }

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

constructor CTuningMap.Create;
begin
 DecimalSeparator:='.';
 Reset;
end;

destructor CTuningMap.Destroy;
begin
end;

//////////////////////////////////////////////////////////////////////
// Public Functions
//////////////////////////////////////////////////////////////////////

procedure CTuningMap.Reset;
var i:integer;
begin
 // This function _must_ never produce an error, so we don't need
 // to return a bool value...
 m_szErrorString:='';
 m_dblBaseFreq:=8.1757989156437073336; // Means A = 440Hz
 for i:=0 to 127 do m_dblTunes[i]:=100*i;
end;

function CTuningMap.WriteToFile(szFilepath: string;
  bSaveBaseFreq: boolean): boolean;
var i:integer;
    ofs:TFileStream;
    s:string;
const endl=#13#10;
begin
 m_szErrorString:='';
 ofs:=TFileStream.create(szFilePath,fmCreate);
 s:=';'+endl;ofs.Write(s[1],length(s));
 s:='; AnaMark / VAZ 1.5 Plus tuning map file'+endl;ofs.Write(s[1],length(s));
 s:=';'+endl;ofs.Write(s[1],length(s));
 s:=';'+endl;ofs.Write(s[1],length(s));
 s:='; 1. VAZ-section with quantized tunings'+endl;ofs.Write(s[1],length(s));
 s:=';'+endl;ofs.Write(s[1],length(s));
 s:='[Tuning]'+endl;ofs.Write(s[1],length(s));

 for i:=0 to 127 do
 begin
  s:='note '+inttostr(i)+' = '+inttostr(round(m_dblTunes[i]))+endl;
  ofs.Write(s[1],length(s));
 end;

 s:=';'+endl;ofs.Write(s[1],length(s));
 s:='; 2. AnaMark-specific section with exact tunings'+endl;ofs.Write(s[1],length(s));
 s:=';'+endl;ofs.Write(s[1],length(s));
 s:='[Exact Tuning]'+endl;ofs.Write(s[1],length(s));

 if (bSaveBaseFreq) then
 begin
  s:='basefreq = '+floattostr(m_dblBaseFreq)+endl;
  ofs.Write(s[1],length(s));
 end;

 for i:=0 to 127 do
 begin
  s:='note '+inttostr(i)+' = '+floattostr(round(m_dblTunes[i]))+endl;
  ofs.Write(s[1],length(s));
 end;

 ofs.free;
 // Always returns true, because currently there's no error processing
 result:=true;
end;

function CTuningMap.ReadFromFile(szFilepath: string): boolean;
var
    secCurr:eSection;
    lTunes:array[0..127] of longint; // For temporary use, unit: Cents
    bTuningFound,bExactTuningFound:boolean;
    lNoteIndex,nCurrPos,i,j,lET_LastNoteFound,lLineCount:longint;
    ifs:TFileStream;
    pstr:array[0..255] of char;
    ch:char;
    p:double;
    szParam,szValue,szLine,szCurr:string;
begin
 secCurr:=SEC_None;
 bTuningFound:=false;
 bExactTuningFound:=false;
 lET_LastNoteFound:=-1;
 lLineCount:=0;

 // Initialize data
 // Important, because notes not listed in the tuning file
 // should always have standard tuning.
 Reset;
 for i:=0 to 127 do lTunes[i]:=round(m_dblTunes[i]);

 // Now open the file
 ifs:=TFileStream.Create(szFilePath,fmOpenRead or fmShareDenyNone);
 if not assigned(ifs) then
 begin
  m_szErrorString:='Error opening the file: '+szFilepath;
  strpcopy(pstr,m_szErrorString);messagebox(0,pstr,'Error',1);
  result:=false;
  exit;
 end;

 while (ifs.position<ifs.size) do
 begin
  // Increase Line counter to make it easier detecting errors, if
  // a more detailed output is wanted.
  inc(lLineCount);

  // Read line, until '\n', '\r' or '\0' is reached
  // Thus it is able to read WIN/DOS-style as well as UNIX-style files
  // By the way: Skip empty lines or multiple line-end-characters
  // Is not case sensitive, so all chars are converted to lower ones
  nCurrPos:=0;szLine:='';
  repeat
   while ( (ifs.position<ifs.size) and (nCurrPos < 510) ) do
   begin
    ch:=' ';
    ifs.read(ch,1);
    if ( (ch=#0) or (ch=#10) or (ch=#13) ) then break;
    szLine:=szLine+ch;
    inc(nCurrPos);
   end
  until ( (ifs.position>=ifs.size) or (nCurrPos <> 0) );
  if ( nCurrPos >= 510 ) then
  begin
   ifs.free;
   m_szErrorString:='Line too long (line '+inttostr(lLineCount)+')';
   strpcopy(pstr,m_szErrorString);messagebox(0,pstr,'Error',1);
   result:=false; // Line too long
   exit;
  end;
  if szLine='' then continue;

  szLine:=lowercase(szLine);
  szLine[nCurrPos+1]:=#0;

  szCurr:=szLine;              

  // Skip empty lines
  if ( szCurr = #0 ) then continue;

  // Skip leading and trailing spaces/tabs
  szCurr:=StripBlanks(szCurr);

  // Skip comment lines
  if ( szCurr[1] = ';' ) then continue;

  // Check for new section
  if ( szCurr[1] = '[' ) then
  begin
   if ( szCurr[length(szCurr)] <> ']' ) then
   begin
    ifs.free;
    m_szErrorString:='Syntax error: Section-tag must be the only string in the line! (line '+inttostr(lLineCount)+')';
    strpcopy(pstr,m_szErrorString);messagebox(0,pstr,'Error',1);
    result:=false; // error in section-tag! Must be the only one string in the line!
    exit;
   end;
   // Known section found?
   secCurr:=SEC_Unknown;
   if (szCurr='[tuning]') then
   begin
    secCurr:=SEC_Tuning;
    bTuningFound:=true;
   end;
   if (szCurr='[exact tuning]') then
   begin
    secCurr:=SEC_ExactTuning;
    bExactTuningFound:=true;
   end;

   // Now process next line
   continue;
  end;

  // Skip all lines which are in none or in an unknown section
  if ( (secCurr=SEC_None) or (secCurr=SEC_Unknown) ) then continue;

  // Separate parameter name and value
  j:=pos('=',szCurr);
  if j<1 then
  begin
   ifs.free;
   m_szErrorString:='Syntax error: "=" missing! (line '+inttostr(lLineCount)+')';
   strpcopy(pstr,m_szErrorString);messagebox(0,pstr,'Error',1);
   result:=false; // definitely an error: '=' missing!
   exit;
  end;
  szParam:=copy(szCurr,1,j-1);
  szValue:=copy(szCurr,j+1,length(szCurr)-j);

  // Skip leading and trailing spaces/tabs
  szParam:=StripBlanks(szParam);
  szValue:=StripBlanks(szValue);

  // Now process the different sections:
  case secCurr of
  SEC_Tuning:
  begin
   // Check for note-tag
   if (copy(szParam,1,4)='note') then
   begin
    // Get MIDI-Note number
    lNoteIndex:=strtoint(copy(szParam,5,length(szParam)-4));
    // Check for correct range [0;127] and ignore it, if it's out of range.
    if ( (lNoteIndex >= 0) and (lNoteIndex <= 127) ) then
     lTunes[lNoteIndex]:=strtoint(szValue);
   end;
  end;
  SEC_ExactTuning:
  begin
   // Check for note-tag
   if (copy(szParam,1,4)='note') then
   begin
    // note-tag found
    // Get MIDI-Note number
    lNoteIndex:=strtoint(copy(szParam,5,length(szParam)-4));
    // Check for correct range [0;127] and ignore it, if it's out of range.
    if ( (lNoteIndex >= 0) and (lNoteIndex <= 127) ) then
     m_dblTunes[lNoteIndex]:=strtofloat(szValue);

    if ( lET_LastNoteFound < lNoteIndex ) then lET_LastNoteFound:=lNoteIndex;
   end;
   // Check for basefreq parameter
   if (copy(szParam,1,8)='basefreq') then
   begin
    // basefreq found
    m_dblBaseFreq:=strtofloat(szValue);
   end;
  end;
  end;
 end;

 if ( (not bTuningFound) and (not bExactTuningFound) ) then
 begin
  ifs.free;
  m_szErrorString:='No tuning data found!';
  strpcopy(pstr,m_szErrorString);messagebox(0,pstr,'Error',1);
  result:=false; // No tuning data found at all should be worth an error...
  exit;
 end;

 if (not bExactTuningFound) then
 begin
  // There are no exact tuning values, so map the quantized
  // values to the exact ones:
  for i:=0 to 127 do m_dblTunes[i]:=lTunes[i];
 end else
 begin
  // [Exact Tuning] section found, so ignore the values found
  // in the [Tuning] section and do the "auto expand":
  if ( (lET_LastNoteFound >= 0) and (lET_LastNoteFound < 127) ) then
  begin
   // Now loop the given data (auto expand):
   j:=lET_LastNoteFound; // Highest MIDI note number
   P:=m_dblTunes[j];     // Period length
   for i:=j to 127 do m_dblTunes[i]:=m_dblTunes[i-j]+P;
  end;
 end;
 ifs.free;
 result:=true; // Everything nice!
end;

function CTuningMap.GetBaseFreq: double;
begin
 result:=m_dblBaseFreq;
end;

function CTuningMap.GetNoteFreq(nNoteIndex: integer): double;
begin
 result:=GetBaseFreq*power(2,GetRelativeTune(nNoteIndex)/1200);
end;

function CTuningMap.GetRelativeTune(nNoteIndex: integer): double;
begin
 m_szErrorString:='';
 // First make sure, that the note index is in the valid range
 // If not, return a "standard value"
 if ( (nNoteIndex >= 0) and (nNoteIndex <= 127) ) then
  result:=m_dblTunes[nNoteIndex]
 else
  result:=100*nNoteIndex;
end;

function CTuningMap.SetBaseFreq(dblBaseFreq: double): boolean;
var pstr:array[0..255] of char;
begin
 m_szErrorString:='';
 // First make sure, that the base frequency is in the valid range
 // If not, return false;
 if ( dblBaseFreq > 0 ) then
 begin
  m_dblBaseFreq:=dblBaseFreq;
  result:=true;
 end else
 begin
  m_szErrorString:='Base frequency out of range: '+floattostr(dblBaseFreq);
  strpcopy(pstr,m_szErrorString);messagebox(0,pstr,'Error',1);
  result:=false;
 end;
end;

function CTuningMap.SetRelativeTune(nNoteIndex: integer;
  dblTune: double): boolean;
var pstr:array[0..255] of char;
begin
 m_szErrorString:='';
 // First make sure, that the note index is in the valid range
 // If not, return false;
 if ( (nNoteIndex >= 0) and (nNoteIndex <= 127) ) then
 begin
  m_dblTunes[nNoteIndex]:=dblTune;
  result:=true;
 end else
 begin
  m_szErrorString:='Note index out of range: '+inttostr(nNoteIndex);
  strpcopy(pstr,m_szErrorString);messagebox(0,pstr,'Error',1);
  result:=false;
 end;
end;

function CTuningMap.GetLastError: string;
begin
 result:=m_szErrorString;
end;

end.
